/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY                          *
 *                                                                            *
 * This program is free software; you can redistribute it and/or modify       *
 * it under the terms of the GNU General Public Liense as published by        *
 * the Free Software Foundation, either version 2 of the License, or (at      * 
 * your option) any later version.                                            *
 *                                                                            *
 * The ITX package is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY, without even the implied warranty of MERCHANTABILITY *
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
 * for more details.                                                          * 
 *                                                                            *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.                                                   *
 *                                                                            * 
 * Contact information:                                                       *
 * Donna Bergmark                                                             *
 * 484 Rhodes Hall                                                            *
 * Cornell University                                                         *
 * Ithaca, NY 14853-3801                                                      *
 * bergmark@cs.cornell.edu                                                    *
 ******************************************************************************/
package shared;

import java.io.*;
import java.util.Vector;

/**
 * A <code>PAM</code> is a Presentation to Audio Mapping.  It is used to store
 * information about a presentation.  Specifically, it contains discriptor
 * entries which map a presentation slide to its corresponding PowerPoint
 * presentation slide and audio stream offset.  Furthermore, a PAM contains
 * topic index entries which map presentation slides to topic strings.
 * 
 * @author Jason Howes
 * @version 1.0, 2/21/1999
 * @see cnrg.apps.spot.shared.PAMDescriptorEntry
 * @see cnrg.apps.spot.shared.PAMTopicIndexEntry
 * @see cnrg.apps.spot.shared.PAMViewer
 */
public class PAM implements Serializable
{
	/**
	 * <code>PAM</code> fields.
	 */
	private String mInfo;
	private int mNumPresentationSlides;
	private Vector mTopicIndexEntries;
	private Vector mDescriptorEntries;

	/**
	 * Class constructor.
	 */
	public PAM()
	{
		mTopicIndexEntries = new Vector();
		mDescriptorEntries = new Vector();
		mNumPresentationSlides = 0;
	}

	/**
	 * Prints the contents of a <code>PAM</code>
	 * 
	 * @exception <code>ArrayIndexOutOfBoundsException</code> on error.
	 */
	public void print() throws ArrayIndexOutOfBoundsException
	{
		int size;
		PAMTopicIndexEntry ptie;
		PAMDescriptorEntry pde;

		System.out.println("---------------------");
		System.out.println("mInfo: " + mInfo);
		System.out.println("mNumPresentationSlides: " + mNumPresentationSlides);

		size = getNumTopicIndexEntries();
		System.out.println("mTopicIndexEntries: ");
		for (int i = 0; i < size; i++)
		{
			ptie = topicIndexEntryAt(i);
			System.out.println("\tmTopic: " + ptie.mTopic);
			System.out.println("\tmPresentationSlideNum: " + ptie.mPresentationSlideNum);
		}

		size = getNumDescriptorEntries();
		System.out.println("mDescriptorEntries: ");
		for (int i = 0; i < size; i++)
		{
			pde = descriptorEntryAt(i);
			System.out.println("\tmPPTSlideNum: " + pde.mPPTSlideNum);
			System.out.println("\tmPresentationSlideNum: " + pde.mPresentationSlideNum);
			System.out.println("\tmRADOffset: " + pde.mRADOffset);
		}

		System.out.println("---------------------");
	}

	/**
	 * Returns the <code>PAM</code> information string.
	 * 
	 * @return the <code>PAM</code> information string.
	 */
	public String getInfo()
	{
		return mInfo;
	}

	/**
	 * Sets the <code>PAM</code> information string.
	 */
	public void setInfo(String info)
	{
		mInfo = new String(info);
	}

	/**
	 * Gets the number of mapped presentation slides.
	 * 
	 * @return the number of presentation slides.
	 */
	public int getNumPresentationSlides()
	{
		return mNumPresentationSlides;
	}

	/**
	 * Gets the number of topic index entries.
	 * 
	 * @return the number of topic index entries.
	 */
	public int getNumTopicIndexEntries()
	{
		return mTopicIndexEntries.size();
	}

	/**
	 * Gets the number of descriptor entries.
	 * 
	 * @return the number of descriptor entries.
	 */
	public int getNumDescriptorEntries()
	{
		return mDescriptorEntries.size();
	}

	/**
	 * Adds a topic index entry.
	 * 
	 * @param topicIndexEntry the topic index entry to add.
	 */
	public void addTopicIndexEntry(PAMTopicIndexEntry topicIndexEntry)
	{
		mTopicIndexEntries.addElement(topicIndexEntry);
	}

	/**
	 * Inserts a topic index entry at a specified offset.
	 * 
	 * @param topicIndexEntry the topic index entry to add.
	 * @param index the index where the new topic index entry will be inserted.
	 * @exception <code>ArrayIndexOutOfBoundsException</code> on invalid index.
	 */
	public void insertTopicIndexEntryAt(PAMTopicIndexEntry topicIndexEntry,
										int index) throws ArrayIndexOutOfBoundsException
	{
		mTopicIndexEntries.insertElementAt(topicIndexEntry, index);
	}

	/**
	 * Retrieves a topic index entry at a specified offset.
	 * 
	 * @param index the index where the new topic index entry will be retrieved from.
	 * @return the specified topic index entry.
	 * @exception <code>ArrayIndexOutOfBoundsException</code> on invalid index.
	 */
	public PAMTopicIndexEntry topicIndexEntryAt(int index) throws ArrayIndexOutOfBoundsException
	{
		return (PAMTopicIndexEntry)mTopicIndexEntries.elementAt(index);
	}
	
	/**
	 * Returns a <code>Vector</code> containing <code>String</code> descriptions of the topics.
	 * 
	 * @return a <codde>Vector</code> of topics.
	 */
	public Vector getTopics()
	{
		Vector returnVector = new Vector();
		PAMTopicIndexEntry currentEntry;
		int numTopicEntries = getNumTopicIndexEntries();

		// Loop through all topics, constructing topic Vector?
		for (int i = 0; i < numTopicEntries; i++)
		{
			try
			{
				currentEntry = topicIndexEntryAt(i);
				returnVector.addElement(new String(currentEntry.mTopic));
			}
			catch (Exception e)
			{
				break;
			}
		}
		
		return returnVector;
	}

	/**
	 * Removes a topic index entry at a specified offset.
	 * 
	 * @param index the index where the new topic index entry will be removed from.
	 * @exception <code>ArrayIndexOutOfBoundsException</code> on invalid index.
	 */
	public void removeTopicIndexEntryAt(int index) throws ArrayIndexOutOfBoundsException
	{
		mTopicIndexEntries.removeElementAt(index);
	}

	/**
	 * Removes all topic index entries.
	 */
	public void removeAllTopicIndexEntries()
	{
		mTopicIndexEntries.removeAllElements();
	}

	/**
	 * Adds a descriptor entry.
	 * 
	 * @param descriptorEntry the descriptor entry to add.
	 */
	public void addDescriptorEntry(PAMDescriptorEntry descriptorEntry)
	{
		mDescriptorEntries.addElement(descriptorEntry);
		mNumPresentationSlides++;
	}

	/**
	 * Inserts a descriptor entry at a specified offset.
	 * 
	 * @param descriptorEntry the descriptor entry to add.
	 * @param index the index where the new descriptor entry will be inserted.
	 * @exception <code>ArrayIndexOutOfBoundsException</code> on invalid index.
	 */
	public void insertDescriptorEntryAt(PAMDescriptorEntry descriptorEntry,
										int index) throws ArrayIndexOutOfBoundsException
	{
		mDescriptorEntries.insertElementAt(descriptorEntry, index);
		mNumPresentationSlides++;
	}

	/**
	 * Retrieves a descriptor entry at a specified offset.
	 * 
	 * @param index the index where the descriptor entry will be retrieved from.
	 * @return the specified descriptor entry.
	 * @exception <code>ArrayIndexOutOfBoundsException</code> on invalid index.
	 */
	public PAMDescriptorEntry descriptorEntryAt(int index) throws ArrayIndexOutOfBoundsException
	{
		return (PAMDescriptorEntry)mDescriptorEntries.elementAt(index);
	}

	/**
	 * Removes a descriptor entry at a specified offset.
	 * 
	 * @param index the index where the descriptor entry will be removed from.
	 * @exception <code>ArrayIndexOutOfBoundsException</code> on invalid index.
	 */
	public void removeDescriptorEntryAt(int index) throws ArrayIndexOutOfBoundsException
	{
		mDescriptorEntries.removeElementAt(index);
		mNumPresentationSlides--;
	}
	
	/**
	 * Removes all descriptor entries.
	 */
	public void removeAllDescriptorEntries()
	{
		mDescriptorEntries.removeAllElements();
		mNumPresentationSlides = 0;
	}
}
